// --------------------------------- lda_definitions.h -------------------------------------------
//
//  Include file for Linux LabBrick attenuator defintions
//
//  (c) 2020-2024 by Vaunix Technology Corporation, all rights reserved
//
//  JA  Version 1.0 LDA Ethernet Socket Device Driver
//  RD	5/12/24 Fixes and updates
//  RD  7/13/24 Added a parameter to track the number of channels in a base module
//
//-----------------------------------------------------------------------------
#ifndef __LDASOCKET_H__
#define __LDASOCKET_H__

/// ---------- Include headers ----------------
#include "ldadrvr.h"

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif
/// ---------- Macros ----------------

// DWELL Time
#define VNX_MIN_DWELLTIME 1
#define STATUS_PROFILE_ACTIVE 0x80	// MASK: A profile is playing

// HID Datablock size
#define HR_BLOCKSIZE 6				// size of the block of bytes buffer in our report

// HID Command Message Length
#define HID_REPORT_LENGTH 8         // we use an 8 byte report..

// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)
#define SWP_DIRECTION       0x04	// MASK: bit = 0 for sweep or ramp up, 1 for sweep or ramp down
#define SWP_CONTINUOUS      0x02	// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE            0x01	// MASK: bit = 1 for single sweep
#define SWP_BIDIR           0x10	// MASK: bit = 0 for ramp style sweep, 1 for triangle style sweep

// ----------- Mode Bit Masks ------------
#define MODE_RFON       0x00000100  // bit is 1 for RF on, 0 if RF is off
#define MODE_SWEEP      0x0000001F  // bottom 5 bits are used to keep the ramp control bits (8 are reserved)

// LDA command message ID to send to the device
#define VNX_GET         0x00    	// the set and get bits are or'd into the msb of the command byte
#define VNX_SET         0x80

// ---------------------- Attenuator commands ------------------------
#define VNX_PWR         0x0D    // power output setting, relative to calibrated value - adds to calibrated
                                // attenuator setting. It is a byte, with the attenuation expressed in HW
                                // specific steps.

#define VNX_FREQUENCY   0x04    // working frequency in 100Khz units
// ----------------- Attenuator ramp commands ------------------------
#define VNX_SWEEP       0x09    // command to start/stop sweep, data = 01 for single sweep, 00 to stop
                                // sweeping, and 02 for continuous sweeping.

#define VNX_RFMUTE      0x0A    // enable or disable RF output, byte = 01 to enable, 00 to disable

#define VNX_ASTART      0x30    // initial value for attenuation ramp

#define VNX_ASTOP       0x31    // final value for attenuation ramp

#define VNX_ASTEP       0x32    // step size for attenuation ramp
#define VNX_ASTEP2      0x38    // step size for the second phase of the ramp

#define VNX_ADWELL      0x33    // dwell time for each attenuation step
#define VNX_ADWELL2     0x37    // dwell time for the second phase of the ramp

#define VNX_AIDLE       0x36    // idle time between attenuation ramps in milliseconds
#define VNX_AHOLD       0x39    // hold time between phase 1 and 2

#define VNX_SETPROFILE  0x3A    // set/get profile values, first byte is unused
                                // second data byte is the index (0 based) (for non-HiRes devices)
                                // the third is the attenuation value for that profile entry

#define VNX_PROFILECOUNT    0x3B    // number of elements in the profile, 1 to PROFILE_MAX = 100
#define VNX_PROFILEDWELL    0x3C    // dwell time for each profile element
#define VNX_PROFILEIDLE     0x3D    // idle time at the end of each repeating profile

#define VNX_SAVEPAR     0x0C    // command to save user parameters to flash, data bytes must be
                                // set to 0x42, 0x55, 0x31 as a key to enable the flash update
                                // all of the above settings are saved (RF Mute State, Attenuation,
                                // sweep parameters, etc.

#define VNX_MINATTEN    0x34    // get the minimum attenuation level which is 0 for every case now

#define VNX_MAXATTEN    0x35    // get the maximum attenuation level in .05 db units

#define VNX_GETSERNUM   0x1F    // get the serial number, value is a DWORD

#define VNX_MODELNAME   0x22    // get (no set allowed) the device's model name string -- last 6 chars only

#define VNX_DEFAULTS    0x0F    // restore all settings to factory default
                                // ASTART = 0 = MINATTEN, ASTOP = MAXATTEN
                                // ADWELL = 1000 = 1 second, ASTEP = 2 = .5db, AIDLE = 0

// ------------------------ Hi Res Attenuator Commands --------------------------------
#define VNX_MINFREQUENCY    0x20    // get (no set allowed) the device's minimum working frequency
#define VNX_MAXFREQUENCY    0x21    // get (no set allowed) the device's maximum working frequency

//------------------------- Quad Attenuator Commands-----------------------------------
#define VNX_CHANNEL         0x54    // set the channel
//------------------------- Status Report ID Byte -------------------------------------
#define VNX_STATUS      0x0E    // Not really a command, but the status byte value for periodic status reports.
#define VNX_HRSTATUS    0x52    // status report used by HiRes (1 and 4 channel)
#define VNX_HR8STATUS   0x55    // status report used by HiRes 8 channel devices
#define VNX_MAXCHANNEL  0x56    // Report the Maxchannels of the device

// ---------------------- LDA system Specific Commands ---------------------
#define VNX_SWVERSION      0x60	// SW Version
#define VNX_IPMODE         0x61	// IP interface mode Static/DHCP
#define VNX_IPADDR         0x62	// IP Address
#define VNX_IPMASK         0x63	// Subnet Mask
#define VNX_IPGW           0x64	// Gateway

// ----------- Global Equates ------------
#define MAXDEVICES 64			// I changed to non zero devid values,
								// the array is one element larger so all active devid values are non-zero
#define PROFILE_MAX 100         // Older non-hires attenuators can store 100 profile elements in their eeprom
#define PROFILE_MAX_RAM 1000    // New FW V2.x based attenuators have a 1000 element RAM profile buffer
#define PROFILE_MAX_HR 50       // HiRes Attenuators only save 50 elements in their eeprom
#define CHANNEL_MAX 64          // Expandable devices can have, in theory, up to 64 channels

// ----------- Data Types ----------------
#define DEVID unsigned int
#define PROFILE_EMPTY 0xFFFF    // we use this as our marker for an empty entry in the profile (which is an unsigned short)

// Device IP length (IPV4)
#define DEVICE_IPV4_LENGTH 16
#define RXBUFF_DATA_LENGTH 24
#define TXBUFF_CMD_LENGTH   8

// ----------- Profile Control -----------
#define PROFILE_ONCE    1           // play the profile once
#define PROFILE_REPEAT  2           // play the profile repeatedly
#define PROFILE_OFF     0           // stop the profile

// ----------- Command Equates -----------
#define BYTE unsigned char

// Response Data Message Datastructure
typedef struct
{
  char command;
  char length;
  char data[HR_BLOCKSIZE];
} RESPONSE_DATA_T;

// LDA Device data Structure
// RD 5/12/24 - added a field for a flag to indicate we have determined the number of channels for
//				the device.

typedef struct
{
  //  Global device variables
  int DevType;
  int MinFrequency;
  int MaxFrequency;
  int MinAttenuation;
  int MaxAttenuation;           // maximum attenuation in .05 db units
  int MinAttenStep;             // replaces DevResolution, smallest attenuation step in .05 db units
  int UnitScale;                // size of hardware unit in .05 db units ( .25db = 5, .05db = 1. .5db = 10, 1db = 20)
  int NumChannels;              // the number of channels the device has
  int ProfileMaxLength;         // 50, 100, or 1000 depending on the device
  int ProfileMaxSaved;          // the maximum length of the profile that can be saved in eeprom, 50 for HiRes devices, 100 for other devices
  volatile int Channel;         // the current channel number - zero based
  volatile int ChMask;          // our channel in bitmask form (0x01 to 0x80 for an 8 channel logical bank)
  volatile int ChBank;			// the logical bank for the current channel, zero based
  int BaseChannels;				// the number of channels in the base unit
  int SerialNumber;
  char ModelName[MAX_MODELNAME];
  char Swversion[MAX_SWVERSION];
  char HaveNumChannels;			// set to a non-zero value if we know the number of channels in the LDA device
  int  ipmode;
  char ipaddress[MAX_NETBUFF];
  char netmask[MAX_NETBUFF];
  char gateway[MAX_NETBUFF];
  //  Per channel variables
  volatile int WorkingFrequency[CHANNEL_MAX];
  volatile int Attenuation[CHANNEL_MAX];                // in .05db units
  volatile int RampStart[CHANNEL_MAX];
  volatile int RampStop[CHANNEL_MAX];
  volatile int AttenuationStep[CHANNEL_MAX];            // ramp step size for the first phase of the ramp
  volatile int AttenuationStep2[CHANNEL_MAX];           // ramp step size for second phase of the ramp
  volatile int DwellTime[CHANNEL_MAX];
  volatile int DwellTime2[CHANNEL_MAX];
  volatile int IdleTime[CHANNEL_MAX];
  volatile int HoldTime[CHANNEL_MAX];
  volatile int ProfileCount[CHANNEL_MAX];
  volatile int ProfileDwellTime[CHANNEL_MAX];
  volatile int ProfileIdleTime[CHANNEL_MAX];
  volatile int Modebits[CHANNEL_MAX];
  volatile unsigned short Profile[CHANNEL_MAX][PROFILE_MAX_RAM];    // this allocates 8000 ints for each of 64 devices, we try to shrink it by only allocating Uint16 elements
                                                        // we don't use a malloc/free strategy to avoid memory leak issues if an application fails to close a device
  // Internal variables used to identify and manage the hardware
  char deviceip[DEVICE_IPV4_LENGTH];
  char sndbuff[TXBUFF_CMD_LENGTH];
  char rcvbuff[RXBUFF_DATA_LENGTH];
  int  devicesockfd;
  int  deviceready; // Set the device ready flag 
  int  responseready; // Response ready flag
} LDAPARAMS;

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __LDASOCKET_H__
